/* 
  Inline-Edit für Elemente mit .editable-click
  - Klick ersetzt das Element durch ein Edit-Feld mit Save/Cancel
  - Enter speichert
  - Esc bricht ab
  - Keine Inline-onclick-Handler, kein Base64, keine Umlautprobleme
*/

$(function () {
    // Event Delegation: funktioniert auch nach replaceWith, ohne neu zu binden
    $(document).on("click", ".editable-click", editableBind);

    // Button-Handler im Edit-Modus
    $(document).on("click", ".editable-area .js-cancel", editableCancel);
    $(document).on("click", ".editable-area .js-save", editableSave);

    // Keyboard-Handler im Edit-Modus: Enter = Save, Esc = Cancel
    $(document).on("keydown", ".editable-area input[type='text']", editableKeydown);
});

function editableBind() {
    const $elem = $(this);

    // Pflicht: das editierbare Element braucht eine id und data-url
    const id = $elem.attr("id");
    const postURL = $elem.data("url"); // liest data-url
    if (!id || !postURL) return;

    const textboxID = "Textfeld_" + id;
    const areaID = "EditableArea_" + id;

    // Aktuellen Anzeige-Text übernehmen
    const text = $elem.text();

    // Breite grob übernehmen (mit Untergrenze)
    const width = Math.max(50, parseFloat($elem.outerWidth()) - 50);

    // Original-HTML des Elements sichern (Unicode-sicher als normaler String)
    const originalHTML = document.getElementById(id).outerHTML;

    // Edit-UI bauen (kein value= im HTML-String, Wert wird danach per .val gesetzt)
    // Hinweis: class="btn-group" (nicht "clas")
    const $area = $(`
    <div id="${areaID}" class="editable-area" style="display:inline-block; min-width:${width}px;">
      <div class="input-group">
        <input type="text" id="${textboxID}" class="form-control" style="min-width:${width}px">
        <div class="btn-group" style="margin-left:2px;">
          <button type="button" class="btn btn-outline-primary js-save">
            <div id="${textboxID}_Spinner" class="spinner-border spinner-border-sm" style="display:none;" role="status"></div>
            <span id="${textboxID}_Check" class="fa fa-check"></span>
          </button>
          <button type="button" class="btn btn-outline-danger js-cancel">
            <span class="fa fa-xmark"></span>
          </button>
        </div>
      </div>
    </div>
  `);

    // Metadaten am Container speichern, damit Save/Cancel sie später findet
    $area.attr("data-original-html", originalHTML);
    $area.attr("data-post-url", postURL);
    $area.attr("data-element-id", id);
    $area.attr("data-textbox-id", textboxID);

    // Input-Wert sicher setzen
    $area.find("#" + textboxID).val(text);

    // Original ersetzen
    $elem.replaceWith($area);

    // Optional: Fokus setzen und Cursor ans Ende
    const input = document.getElementById(textboxID);
    if (input) {
        input.focus();
        input.setSelectionRange(input.value.length, input.value.length);
    }
}

function editableKeydown(e) {
    // Enter speichert (ohne Shift)
    if (e.key === "Enter" && !e.shiftKey) {
        e.preventDefault();
        $(this).closest(".editable-area").find(".js-save").trigger("click");
        return;
    }

    // Esc bricht ab
    if (e.key === "Escape") {
        e.preventDefault();
        $(this).closest(".editable-area").find(".js-cancel").trigger("click");
        return;
    }
}

function editableCancel() {
    const $area = $(this).closest(".editable-area");
    const originalHTML = $area.attr("data-original-html");

    // Original wiederherstellen
    $area.replaceWith(originalHTML);
}

function editableSave() {
    const $area = $(this).closest(".editable-area");

    const postURL = $area.attr("data-post-url");
    const elementID = $area.attr("data-element-id");
    const textboxID = $area.attr("data-textbox-id");
    const originalHTML = $area.attr("data-original-html");

    const value = $("#" + textboxID).val();

    const $spinner = $("#" + textboxID + "_Spinner");
    const $check = $("#" + textboxID + "_Check");

    // UI: Loading anzeigen
    $spinner.show();
    $check.hide();

    // POST
    $.post(postURL, { value: value })
        .done(function () {
            // Original wiederherstellen
            $area.replaceWith(originalHTML);

            // Text aktualisieren (falls sich der Inhalt geändert hat)
            $("#" + elementID).text(value);
        })
        .fail(function () {
            // Bei Fehler im Edit-Modus bleiben und UI zurücksetzen
            alert("Speichern fehlgeschlagen.");
        })
        .always(function () {
            // Spinner/Check zurücksetzen (wenn Area noch existiert)
            $spinner.hide();
            $check.show();
        });
}
